/****************************************************************
 
	freeengl.c - the free English spell checker
 
	=============================================================
  Copyright 1996-2005 Tom Barbalet. All rights reserved.
  
  This is an open source license. Please contact Tom Barbalet 
  <tom at nobleape dot com> for commercial licensing options.
  
  Redistribution and use in source and binary forms, with or 
  without modification, are permitted provided that the 
  following conditions are met:
  
    1. Redistributions of source code must retain the above 
  copyright notice, this list of conditions and the following 
  disclaimer. 
    2. Redistributions in binary form must reproduce the 
  above copyright notice, this list of conditions and the 
  following disclaimer in the documentation and/or other 
  materials provided with the distribution.
    3. Redistributions in any form must be accompanied by 
  information on how to obtain complete source code for this 
  software and any accompanying software that uses this 
  software. The source code must either be included in the 
  distribution or be available for no more than the cost of 
  distribution plus a nominal fee, and must be freely 
  redistributable under reasonable conditions. For an executable 
  file, complete source code means the source code for all modules 
  it contains. It does not include source code for modules or 
  files that typically accompany the major components of the 
  operating system on which the executable file runs.
  
  THIS SOFTWARE IS PROVIDED BY TOM BARBALET "AS IS" AND ANY 
  EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED 
  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL TOM 
  BARBALET, NOBLE APE OR ITS CONTRIBUTORS BE LIABLE FOR ANY 
  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
  AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT 
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  
  =============================================================
  
  This software and Noble Ape are a continuing work of Tom 
  Barbalet, begun on 13 June 1996. No apes or cats were harmed 
  in the writing of this software.
	
****************************************************************/


#include <stdio.h>
#include <stdlib.h>

/*
	span_node points to an array of character_node(s)

	the character_node can then point to an additional span_node

	to identify the end of a string, the 0 value is used
 */

typedef	struct{
	unsigned char	value;
	void *          next_character;
}character_node;

typedef struct{
	unsigned char	number;
	character_node  *span;
}span_node;

/*
	if more span space is needed, rather than discarding the memory location,
	the span allocation based on size, is filed and reused in a link list...
 */

typedef	struct{
	character_node   *span;
	void *           next_span;
}span_mm;

#define WHERES_THE_MEMORY_GOING

#define	DICTNAME			"concise_cmu.txt"
#define	FILENAME			"test_words.txt"

#define	STRING_LENGTH		32
#define	NUMBER_ENTRIES		119651


#define IN_INIT_SPAN_MM			0
#define IN_CHARACTER_NODE		1
#define	IN_SPAN_MM_FREE			2
#define	IN_SPAN_NODE			3

#ifdef	WHERES_THE_MEMORY_GOING

int	malloc_counter[4] = {0};

#endif

static void * counting_malloc(int size, int when){

#ifdef WHERES_THE_MEMORY_GOING
	malloc_counter[ when ] += size;
#endif
	return malloc(size);
}

/*

  initialise the memory management with the number different size deltas to track

 */

static span_mm * init_span_mm(){
	span_mm     *temp = 0L;
	int			loop = 0;
	temp = (span_mm *)counting_malloc(sizeof(span_mm) * STRING_LENGTH, IN_INIT_SPAN_MM);
	if(temp == 0L)
		return 0L;
	while(loop < STRING_LENGTH){
		temp[loop].span = 0L;
		temp[loop].next_span = 0L;
		loop++;
	}
	return temp;
}

/*
	cleans the dead ends along the link list at final clean-up
 */

static void clean_end_span_mm(span_mm ** temp){
	if((*temp) != 0L){
		if((*temp)->span != 0L)
			free((*temp)->span);
		clean_end_span_mm((span_mm **)&((*temp)->next_span));
		free(*temp);
	}
}

/*
	as the array points are allocated differently to the span, the
	clean up needs to be treated with a degree of sensitivity to the 
	array format
 */

static void clean_span_mm(span_mm ** value){
	span_mm *temp = *value;
	int loop = 0;
	while(loop < STRING_LENGTH){
		clean_end_span_mm((span_mm **)&temp[loop].next_span);
		if(temp[loop].span != 0L)
			free(temp[loop].span);
		loop++;
	}
	free(*value);
}

static span_mm * traverse_span_mm(span_mm * local)
{
	int	exit_condition = 1;
	do{
		span_mm * next_test = local->next_span;
		if(next_test == 0L)
			exit_condition = 0;
		else{
			if(next_test->span == 0L)
				exit_condition = 0;
		}
		if(exit_condition)
			local = next_test;
	}while(exit_condition);
	return local;
}

/*
	based on the size of tha allocation needed, the corresponding link list
	is traversed and an available pointer is found and returned.
 */

static character_node * span_mm_malloc(span_mm * mm, unsigned char	length){
	int	         array_location = length;
	span_mm        * local = traverse_span_mm(&mm[ array_location ]);
	character_node * temp = 0L;

	if(local->span == 0L)
	{
		temp = (character_node *) counting_malloc( sizeof(character_node) * length , IN_CHARACTER_NODE);
	}else{
		temp = local->span;
		local->span = 0L;
	}
	return temp;
}

/*
	this is a pseudo free as it really tracks the "deallocated" pointer for
	reuse to avoid lost or speckled memory allocations (if it was traditionally freed).

	some operating systems combat this problem through traditional memory management, it
	is done explicitly here just in case and at no real time penalty.
 */

static int span_mm_free(span_mm * mm, character_node * cn, unsigned char length){
	int	       array_location = length;
	span_mm        * local = traverse_span_mm(&mm[ array_location ]);

	if(local->span == 0L)
		local->span = cn;
	else{
		span_mm *temp = 0L;
		temp = (span_mm *) counting_malloc(sizeof(span_mm), IN_SPAN_MM_FREE);
		if(temp == 0L)
			return -1;
		temp->next_span = 0L;
		temp->span = cn;
		local->next_span = temp;
	}
	return 0;
}


/*

  initialises a clean span node

 */

static span_node * init_span_node(){
	span_node * temp = 0L;
	
	temp = (span_node *) counting_malloc(sizeof(span_node), IN_SPAN_NODE);

	if(temp == 0L)
		return 0L;
	temp->span = 0L;
	temp->number = 0;
	return temp;
}

/*

  adds a character to a span_node, if needed, and returns that character_node or alternative
  if the character_node is already in the span_node, it will just return that character_node

  */

static character_node * init_character_node(span_mm * mm, span_node * temp, unsigned char local_value){

	unsigned char	local_num     =	temp->number;
	character_node  *cn	      = temp->span;
	int		loop          = 0;		
	character_node  *local_cn     = span_mm_malloc(mm, (unsigned char)(local_num+1));
	if(local_cn == 0L)
		return 0L;

	if(local_num != 0){
		while(loop<local_num){
			local_cn[loop].next_character = cn[loop].next_character;
			local_cn[loop].value          = cn[loop].value;
			loop++;
		}
	}
	local_cn[loop].next_character = 0L;
	local_cn[loop].value = 0;
	if(span_mm_free(mm,temp->span,(unsigned char)(temp->number)) == -1){
		return 0L;
	}
	temp->span = local_cn;
	local_cn = &(temp->span[temp->number++]);
	local_cn->value = local_value;
	if(local_value != 0) {
		local_cn->next_character = init_span_node();
	} else {
		local_cn->next_character = 0L;
	}
	return local_cn;
}

/*

  if available, it finds a character_node through a span_node corresponding to
  value. if not available, it returns NULL

  */

static character_node * find_character_node(span_node * start, unsigned char value){

	int	loop = 0;
	int end_loop = start->number;

	while(loop < end_loop){
		character_node * local = start->span;
		if(local[loop].value == value)
			return &local[loop];
		loop++;
	}
	return 0L;
}


/*

	adds a string along a span_node or in the case where mm == 0L, checks the string

 */

static int add_string(span_mm * mm, span_node * start, unsigned char * string){
	int				loop = 0;
	unsigned char	value = string[loop];
	span_node       *current = start;
	do{
		character_node * local_cn;
		value = string[loop++];

		local_cn = find_character_node(current, value);
		if(local_cn == 0L)
		{
			if(mm == 0L)
			{
				return -1;
			}
			local_cn = init_character_node(mm, current, value);
			if(local_cn == 0L)
			{
				return -1;
			}
		}
		current = local_cn->next_character;
	}while(value != 0);
	return 0;
}

/*
	cleans up down the span_nodes
 */

static void clean_span_node(span_node ** value){
	span_node *local_sn = (*value);

	if(local_sn != 0L) {
		int		   end_loop = local_sn->number;
		int		   loop = 0;
		if(end_loop != 0){
			while(loop<end_loop){
				character_node * local_cn = local_sn->span;
				clean_span_node( (span_node **) &( local_cn[loop].next_character ) );			
				loop++;
			}
		}
		if( (*value)->span != 0L)
			free( (*value)->span );
		free(*value);
	}
}

/*
	checks for a valid dictionary entry character - ie something that can be searched
 */

static int	is_dict_char(unsigned char input){
	char	check[6] = "AZ'.";
	if(input == check[3])
		return 1;
	if(input == check[2])
		return 1;
	if(input < check[0])
		return 0;
	if(input > check[1])
		return 0;
	return 1;
}

/*
	sometimes a lingering period (full stop) will show an incorrect spelling erroneously
	this produces a period-less temp_string
 */

static int	remove_final_period(unsigned char * string, unsigned char *temp_string){
	int loop = 0;
	unsigned char value;
	do{
		value = string[loop];
		temp_string[loop++] = value;
	}while(value!=0);
	if(loop == 0)
		return -1;
	if(string[loop-2]!='.')
		return -1;
	temp_string[loop-2] = 0;
	return 0;
}

int main(){
	unsigned char	current_line[STRING_LENGTH] = {0};
	span_node * start = init_span_node();
	span_mm	  * sp_mm = init_span_mm();
	FILE * read_file = fopen(DICTNAME,"rb");
	int	char_counter = 0;
	int	rotate = 0;
	/*
		load the dictionary
	 */

	do{
		unsigned char value;

		fread(&value,1,1,read_file);

		if(value == ','){
			if(add_string(sp_mm, start, current_line) == -1){
				printf("Failed!\n");
				return 0;
			}

			char_counter = 0;
			while(char_counter<STRING_LENGTH){
				current_line[ char_counter++ ] = 0;
			}

			rotate++;
			if((rotate&0x3fff) == 0)
				printf("%d %%\n",((rotate*100)/NUMBER_ENTRIES));
			char_counter = 0;
		}else{
			current_line[char_counter ++] = value;
		}

	}while(!feof(read_file));

	fclose(read_file);

	printf("100 %%\n");
	printf("Success!\n");

	/*
		test text file vs dictionary
	 */


	read_file = fopen(FILENAME,"r");
	char_counter = 0;
	while(char_counter<STRING_LENGTH){
		current_line[ char_counter++ ] = 0;
	}

	char_counter = 0;

	do{
		unsigned char value;

		fread(&value,1,1,read_file);

		if(!(value<'a' || value>'z'))
			value = value - 'a' +'A';

		if(is_dict_char(value)){
			current_line[char_counter ++] = value;
		}else{
			if(add_string(0L, start, current_line) == -1){
				unsigned char	temp_string[STRING_LENGTH]={0};
				int		period_error = remove_final_period(current_line, temp_string);
				if(period_error == 0)
					period_error = add_string(0L, start, temp_string);
				if(period_error == -1){
					printf("%s is spelt incorrectly\n",temp_string);
				}
			}
			char_counter = 0;
			while(char_counter<STRING_LENGTH){
				current_line[ char_counter++ ] = 0;
			}
			char_counter = 0;
		}

	}while(!feof(read_file));
	fclose(read_file);

	clean_span_mm(&sp_mm);
	clean_span_node(&start);

#ifdef WHERES_THE_MEMORY_GOING
	printf("%d\n",malloc_counter[0]);
	printf("%d\n",malloc_counter[1]);
	printf("%d\n",malloc_counter[2]);
	printf("%d\n",malloc_counter[3]);
#endif
	return 1;
}

